Rem CropMarks.vbs
Rem An InDesign VBScript
Rem
Rem Draws crop marks and/or registration marks around the selected object or objects.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem or drop by the InDesign scripting user to user forum at http://www.adobeforums.com.
Rem
main
Function main()
	ReDim myObjectList(0)
	Set myInDesign = CreateObject("InDesign.Application.2024")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll
	If myInDesign.Documents.Count <> 0 Then
	    If myInDesign.Selection.Count <> 0 Then
	        For myCounter = 1 To myInDesign.Selection.Count
	            Select Case TypeName(myInDesign.Selection.Item(myCounter))
	                Case "Rectangle", "Oval", "Polygon", "GraphicLine", "Button", "Group", "TextFrame"
	                    If Not (IsEmpty(myObjectList(0))) Then
	                        ReDim Preserve myObjectList(UBound(myObjectList) + 1)
	                    End If
	                    Set myObjectList(UBound(myObjectList)) = myInDesign.Selection.Item(myCounter)
	            End Select
	        Next
	        If Not (IsEmpty(myObjectList(0))) Then
	            myDisplayDialog myInDesign, myObjectList
	        End If
	    Else
	        MsgBox ("Nothing is selected. Please select an object and try again.")
	    End If
	Else
	    MsgBox ("Please open a document, select an object, and try again.")
	End If
End function
Function myDisplayDialog(myInDesign, myObjectList)
Set myDialog = myInDesign.Dialogs.Add
myDialog.Name = "CropMarks"
With myDialog.DialogColumns.Add
    Set myCropMarksGroup = .EnablingGroups.Add
    myCropMarksGroup.StaticLabel = "Crop Marks"
    myCropMarksGroup.CheckedState = True
    With myCropMarksGroup.DialogColumns.Add
        With .StaticTexts.Add
            .StaticLabel = "Options:"
        End With
    End With
    With myCropMarksGroup.DialogColumns.Add
        With .StaticTexts.Add
            .StaticLabel = "Length:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Offset:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Stroke Weight:"
        End With
    End With
    With myCropMarksGroup.DialogColumns.Add
        Set myCropMarkLengthField = .MeasurementEditboxes.Add
        myCropMarkLengthField.EditUnits = idMeasurementUnits.idPoints
        myCropMarkLengthField.EditValue = 6
        Set myCropMarkOffsetField = .MeasurementEditboxes.Add
        myCropMarkOffsetField.EditUnits = idMeasurementUnits.idPoints
        myCropMarkOffsetField.EditValue = 3
        Set myCropMarkStrokeWeightField = .MeasurementEditboxes.Add
        myCropMarkStrokeWeightField.EditUnits = idMeasurementUnits.idPoints
        myCropMarkStrokeWeightField.EditValue = 0.25
    End With
    Set myRegMarksGroup = .EnablingGroups.Add
    myRegMarksGroup.StaticLabel = "Registration Marks"
    myRegMarksGroup.CheckedState = True
    With myRegMarksGroup.DialogColumns.Add
        With .StaticTexts.Add
            .StaticLabel = "Options:"
        End With
    End With
    With myRegMarksGroup.DialogColumns.Add
        With .StaticTexts.Add
            .StaticLabel = "Inside Target Radius:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Outside Target Radius:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Outside Radius:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Offset:"
        End With
        With .StaticTexts.Add
            .StaticLabel = "Stroke Weight:"
        End With
    End With
    With myRegMarksGroup.DialogColumns.Add
        Set myRegMarkInnerRadiusField = .MeasurementEditboxes.Add
        myRegMarkInnerRadiusField.EditUnits = idMeasurementUnits.idPoints
        myRegMarkInnerRadiusField.EditValue = 2
        Set myRegMarkOuterRadiusField = .MeasurementEditboxes.Add
        myRegMarkOuterRadiusField.EditUnits = idMeasurementUnits.idPoints
        myRegMarkOuterRadiusField.EditValue = 4
        Set myRegMarkRadiusField = .MeasurementEditboxes.Add
        myRegMarkRadiusField.EditUnits = idMeasurementUnits.idPoints
        myRegMarkRadiusField.EditValue = 6
        Set myRegMarkOffsetField = .MeasurementEditboxes.Add
        myRegMarkOffsetField.EditUnits = idMeasurementUnits.idPoints
        myRegMarkOffsetField.EditValue = 3
        Set myRegMarkStrokeWeightField = .MeasurementEditboxes.Add
        myRegMarkStrokeWeightField.EditUnits = idMeasurementUnits.idPoints
        myRegMarkStrokeWeightField.EditValue = 0.25
    End With
    With .BorderPanels.Add
        With .StaticTexts.Add
            .StaticLabel = "Draw Marks Around:"
        End With
        Set myRangeButtons = .RadiobuttonGroups.Add
        With myRangeButtons.RadiobuttonControls.Add
            .StaticLabel = "Each Object"
            .CheckedState = True
        End With
        With myRangeButtons.RadiobuttonControls.Add
            .StaticLabel = "Entire Selection"
        End With
    End With
End With
myReturn = myDialog.Show
If myReturn = True Then
    Rem Get the values from the dialog box.
    myDoCropMarks = myCropMarksGroup.CheckedState
    myDoRegMarks = myRegMarksGroup.CheckedState
    myCropMarkLength = myCropMarkLengthField.EditValue
    myCropMarkOffset = myCropMarkOffsetField.EditValue
    myCropMarkStrokeWeight = myCropMarkStrokeWeightField.EditValue
    myRegMarkInnerRadius = myRegMarkInnerRadiusField.EditValue
    myRegMarkOuterRadius = myRegMarkOuterRadiusField.EditValue
    myRegMarkRadius = myRegMarkRadiusField.EditValue
    myRegMarkOffset = myRegMarkOffsetField.EditValue
    myRegMarkStrokeWeight = myRegMarkStrokeWeightField.EditValue
    myRange = myRangeButtons.SelectedButton
    myDialog.Destroy
    If ((myDoCropMarks <> False) Or (myDoRegMarks <> False)) Then
    Rem Store the current measurement units.
    myOldXUnits = myInDesign.ActiveDocument.ViewPreferences.HorizontalMeasurementUnits
    myOldYUnits = myInDesign.ActiveDocument.ViewPreferences.VerticalMeasurementUnits
    Rem Set the measurement units to points.
    myInDesign.ActiveDocument.ViewPreferences.HorizontalMeasurementUnits = idMeasurementUnits.idPoints
    myInDesign.ActiveDocument.ViewPreferences.VerticalMeasurementUnits = idMeasurementUnits.idPoints
    Rem Draw the printers marks.
        myDrawPrintersMarks myInDesign, myRange, myDoCropMarks, myDoRegMarks, myCropMarkLength, myCropMarkOffset, myCropMarkStrokeWeight, myRegMarkInnerRadius, myRegMarkOuterRadius, myRegMarkRadius, myRegMarkOffset, myRegMarkStrokeWeight, myObjectList
    Rem Reset the measurement units.
    myInDesign.ActiveDocument.ViewPreferences.HorizontalMeasurementUnits = myOldXUnits
    myInDesign.ActiveDocument.ViewPreferences.VerticalMeasurementUnits = myOldYUnits
    Else
        myDialog.Destroy
    End If
End If
End Function

Function myDrawPrintersMarks(myInDesign, myRange, myDoCropMarks, myDoRegMarks, myCropMarkLength, myCropMarkOffset, myCropMarkStrokeWeight, myRegMarkInnerRadius, myRegMarkOuterRadius, myRegMarkRadius, myRegMarkOffset, myRegMarkStrokeWeight, myObjectList)
    Set myDocument = myInDesign.ActiveDocument
    myOldRulerOrigin = myDocument.ViewPreferences.RulerOrigin
    myDocument.ViewPreferences.RulerOrigin = idRulerOrigin.idSpreadOrigin
    Rem Create a layer to hold the printers marks (if it does not already exist).
    Err.Clear
    On Error Resume Next
    Set myLayer = myDocument.Layers.Item("CropMarks")
    If Err.Number <> 0 Then
        Set myLayer = myDocument.Layers.Add
        myLayer.Name = "CropMarks"
        Err.Clear
    End If
    On Error GoTo 0
    Rem Get references to the Registration color, Paper color, and the None swatch.
    Set myRegistrationSwatch = myDocument.Swatches.Item("Registration")
    Set myPaperSwatch = myDocument.Swatches.Item("Paper")
    Set myNoneSwatch = myDocument.Swatches.Item("None")
    Rem Process the objects in the selection.
    For myCounter = 0 To UBound(myObjectList)
        Set myObject = myObjectList(myCounter)
	myBounds = myObject.VisibleBounds
        Rem Set up some initial bounding box values.
        If myCounter = 0 Then
            myX1 = myBounds(1)
            myY1 = myBounds(0)
            myX2 = myBounds(3)
            myY2 = myBounds(2)
        End If
        If myRange = 0 Then
            If myDoCropMarks = True Then
                myDrawCropMarks myInDesign, myBounds(1), myBounds(0), myBounds(3), myBounds(2), myCropMarkLength, myCropMarkOffset, myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
            End If
            If myDoRegMarks = True Then
                myDrawRegMarks myInDesign, myBounds(1), myBounds(0), myBounds(3), myBounds(2), myRegMarkOffset, myRegMarkInnerRadius, myRegMarkOuterRadius, myRegMarkRadius, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myPaperSwatch, myLayer
            End If
        Else
            Rem Compare the bounds values to the stored bounds.
            Rem If a given bounds value is less than (for x1 and y1) or
            Rem greater than (for x2 and y2) the stored value,
            Rem then replace the stored value with the bounds value.
            If myBounds(0) < myY1 Then
                myY1 = myBounds(0)
            End If
            If myBounds(1) < myX1 Then
                myX1 = myBounds(1)
            End If
            If myBounds(2) > myY2 Then
                myY2 = myBounds(2)
            End If
            If myBounds(3) > myX2 Then
                myX2 = myBounds(3)
            End If
        End If
    Next
    If myRange <> 0 Then
            If myDoCropMarks = True Then
                myDrawCropMarks myInDesign, myX1, myY1, myX2, myY2, myCropMarkLength, myCropMarkOffset, myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
            End If
            If myDoRegMarks = True Then
                myDrawRegMarks myInDesign, myX1, myY1, myX2, myY2, myRegMarkOffset, myRegMarkInnerRadius, myRegMarkOuterRadius, myRegMarkRadius, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myPaperSwatch, myLayer
            End If
    End If
    myDocument.ViewPreferences.RulerOrigin = myOldRulerOrigin
End Function
Function myDrawCropMarks(myInDesign, myX1, myY1, myX2, myY2, myCropMarkLength, myCropMarkOffset, myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer)
    Rem Upper left crop mark pair.
    myDrawLine myInDesign, Array(myY1, myX1 - myCropMarkOffset, myY1, myX1 - (myCropMarkOffset + myCropMarkLength)), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myDrawLine myInDesign, Array(myY1 - myCropMarkOffset, myX1, myY1 - (myCropMarkOffset + myCropMarkLength), myX1), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    Rem Lower left crop mark pair.
    myDrawLine myInDesign, Array(myY2, myX1 - myCropMarkOffset, myY2, myX1 - (myCropMarkOffset + myCropMarkLength)), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myDrawLine myInDesign, Array(myY2 + myCropMarkOffset, myX1, myY2 + myCropMarkOffset + myCropMarkLength, myX1), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    Rem Upper right crop mark pair.
    myDrawLine myInDesign, Array(myY1, myX2 + myCropMarkOffset, myY1, myX2 + myCropMarkOffset + myCropMarkLength), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myDrawLine myInDesign, Array(myY1 - myCropMarkOffset, myX2, myY1 - (myCropMarkOffset + myCropMarkLength), myX2), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    Rem Lower left crop mark pair.
    myDrawLine myInDesign, Array(myY2, myX2 + myCropMarkOffset, myY2, myX2 + myCropMarkOffset + myCropMarkLength), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myDrawLine myInDesign, Array(myY2 + myCropMarkOffset, myX2, myY2 + myCropMarkOffset + myCropMarkLength, myX2), myCropMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
End Function

Function myDrawRegMarks(myInDesign, myX1, myY1, myX2, myY2, myRegMarkOffset, myRegMarkInnerRadius, myRegMarkOuterRadius, myRegMarkRadius, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myPaperSwatch, myLayer)
    myXCenter = myX1 + ((myX2 - myX1) / 2)
    myYCenter = myY1 + ((myY2 - myY1) / 2)
    myTargetCenter = myRegMarkOffset + (myRegMarkOuterRadius)
    Rem Top registration target.
    myBounds = Array((myY1 - myTargetCenter) - myRegMarkOuterRadius, myXCenter - myRegMarkOuterRadius, (myY1 - myTargetCenter) + myRegMarkOuterRadius, myXCenter + myRegMarkOuterRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myNoneSwatch, myRegistrationSwatch, myLayer
    myBounds = Array(myY1 - (myTargetCenter + myRegMarkRadius), myXCenter, (myY1 - myTargetCenter) + myRegMarkRadius, myXCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myY1 - myTargetCenter, myXCenter - myRegMarkRadius, myY1 - myTargetCenter, myXCenter + myRegMarkRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array((myY1 - myTargetCenter) - myRegMarkInnerRadius, myXCenter - myRegMarkInnerRadius, (myY1 - myTargetCenter) + myRegMarkInnerRadius, myXCenter + myRegMarkInnerRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array((myY1 - myTargetCenter) - myRegMarkInnerRadius, myXCenter, (myY1 - myTargetCenter) + myRegMarkInnerRadius, myXCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    myBounds = Array(myY1 - myTargetCenter, myXCenter - myRegMarkInnerRadius, myY1 - myTargetCenter, myXCenter + myRegMarkInnerRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    Rem Left registration target.
    myBounds = Array(myYCenter - myRegMarkOuterRadius, (myX1 - myTargetCenter) - myRegMarkOuterRadius, myYCenter + myRegMarkOuterRadius, (myX1 - myTargetCenter) + myRegMarkOuterRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myNoneSwatch, myRegistrationSwatch, myLayer
    myBounds = Array(myYCenter, (myX1 - myTargetCenter) - myRegMarkRadius, myYCenter, (myX1 - myTargetCenter) + myRegMarkRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkRadius, myX1 - myTargetCenter, myYCenter + myRegMarkRadius, myX1 - myTargetCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkInnerRadius, myX1 - (myTargetCenter + myRegMarkInnerRadius), myYCenter + myRegMarkInnerRadius, (myX1 - myTargetCenter) + myRegMarkInnerRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter, (myX1 - myTargetCenter) - myRegMarkInnerRadius, myYCenter, (myX1 - myTargetCenter) + myRegMarkInnerRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkInnerRadius, myX1 - myTargetCenter, myYCenter + myRegMarkInnerRadius, myX1 - myTargetCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    Rem Bottom registration target.
    myBounds = Array((myY2 + myTargetCenter) - myRegMarkOuterRadius, myXCenter - myRegMarkOuterRadius, (myY2 + myTargetCenter) + myRegMarkOuterRadius, myXCenter + myRegMarkOuterRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myNoneSwatch, myRegistrationSwatch, myLayer
    myBounds = Array((myY2 + myTargetCenter) - myRegMarkRadius, myXCenter, (myY2 + myTargetCenter) + myRegMarkRadius, myXCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myY2 + myTargetCenter, myXCenter - myRegMarkRadius, myY2 + myTargetCenter, myXCenter + myRegMarkRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array((myY2 + myTargetCenter) - myRegMarkInnerRadius, myXCenter - myRegMarkInnerRadius, (myY2 + myTargetCenter) + myRegMarkInnerRadius, myXCenter + myRegMarkInnerRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array((myY2 + myTargetCenter) - myRegMarkInnerRadius, myXCenter, (myY2 + myTargetCenter) + myRegMarkInnerRadius, myXCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    myBounds = Array(myY2 + myTargetCenter, myXCenter - myRegMarkInnerRadius, myY2 + myTargetCenter, myXCenter + myRegMarkInnerRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    Rem Right registration target.
    myBounds = Array(myYCenter - myRegMarkOuterRadius, (myX2 + myTargetCenter) - myRegMarkOuterRadius, myYCenter + myRegMarkOuterRadius, (myX2 + myTargetCenter) + myRegMarkOuterRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myNoneSwatch, myRegistrationSwatch, myLayer
    myBounds = Array(myYCenter, (myX2 + myTargetCenter) - myRegMarkRadius, myYCenter, (myX2 + myTargetCenter) + myRegMarkRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkRadius, myX2 + myTargetCenter, myYCenter + myRegMarkRadius, myX2 + myTargetCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkInnerRadius, (myX2 + myTargetCenter) - myRegMarkInnerRadius, myYCenter + myRegMarkInnerRadius, (myX2 + myTargetCenter) + myRegMarkInnerRadius)
    myDrawTarget myInDesign, myBounds, myRegMarkStrokeWeight, myRegistrationSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter, (myX2 + myTargetCenter) - myRegMarkInnerRadius, myYCenter, (myX2 + myTargetCenter) + myRegMarkInnerRadius)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
    myBounds = Array(myYCenter - myRegMarkInnerRadius, myX2 + myTargetCenter, myYCenter + myRegMarkInnerRadius, myX2 + myTargetCenter)
    myDrawLine myInDesign, myBounds, myRegMarkStrokeWeight, myPaperSwatch, myNoneSwatch, myLayer
End Function
Function myDrawLine(myInDesign, myBounds, myStrokeWeight, myStrokeColor, myFillColor, myLayer)
    Set myGraphicLine = myInDesign.ActiveWindow.ActiveSpread.GraphicLines.Add
    myGraphicLine.Move , Array(1, 1)
    myGraphicLine.GeometricBounds = myBounds
    myGraphicLine.ItemLayer = myLayer
    myGraphicLine.StrokeWeight = myStrokeWeight
    myGraphicLine.FillColor = myFillColor
    myGraphicLine.StrokeColor = myStrokeColor
End Function
Function myDrawTarget(myInDesign, myBounds, myStrokeWeight, myFillColor, myStrokeColor, myLayer)
    Set myOval = myInDesign.ActiveWindow.ActiveSpread.Ovals.Add
    myOval.Move , Array(1, 1)
    myOval.ItemLayer = myLayer
    myOval.StrokeWeight = myStrokeWeight
    myOval.FillColor = myFillColor
    myOval.StrokeColor = myStrokeColor
    myOval.GeometricBounds = myBounds
End Function

'' SIG '' Begin signature block
'' SIG '' MIIpyAYJKoZIhvcNAQcCoIIpuTCCKbUCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' 6MBv/fvNrwZyQ2pIfO8MQ3vF2nhAIzmNStu9xiee2bug
'' SIG '' gg6WMIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu
'' SIG '' 2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
'' SIG '' dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAw
'' SIG '' WhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkq
'' SIG '' hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3
'' SIG '' XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH
'' SIG '' 4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAk
'' SIG '' ZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
'' SIG '' ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzh
'' SIG '' GMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRX
'' SIG '' XegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjq
'' SIG '' GzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEj
'' SIG '' R0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
'' SIG '' wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tv
'' SIG '' k2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBe
'' SIG '' M/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7b
'' SIG '' UKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2
'' SIG '' FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrc
'' SIG '' JisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
'' SIG '' 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27Iwy
'' SIG '' CQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQI
'' SIG '' MAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7
'' SIG '' CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/n
'' SIG '' upiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
'' SIG '' BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
'' SIG '' BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQw
'' SIG '' QwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
'' SIG '' aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
'' SIG '' cmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEw
'' SIG '' DQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW
'' SIG '' 4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6j
'' SIG '' fCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3
'' SIG '' WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
'' SIG '' YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGV
'' SIG '' WEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67
'' SIG '' Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIU
'' SIG '' H0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m
'' SIG '' 6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
'' SIG '' 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt
'' SIG '' 2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFS
'' SIG '' i0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+u
'' SIG '' KIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8i
'' SIG '' bEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
'' SIG '' ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+F
'' SIG '' SCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqG
'' SIG '' VnhOMIIH3jCCBcagAwIBAgIQD99ExMbwCe2qSx8hp1o/
'' SIG '' EDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIzMTEw
'' SIG '' MzAwMDAwMFoXDTI1MTEwNDIzNTk1OVowgeYxEzARBgsr
'' SIG '' BgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMI
'' SIG '' RGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3JnYW5p
'' SIG '' emF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYDVQQG
'' SIG '' EwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNhbiBK
'' SIG '' b3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYDVQQL
'' SIG '' EyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29weSwg
'' SIG '' TXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCAiIwDQYJ
'' SIG '' KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPCR5c/cgyg4
'' SIG '' MpJD46/+BWENPhE1RjN77VE2BWlIUdiO2nlWelKthEq9
'' SIG '' aK+1ofKIlDVc9mvxksuXBLESi7HBt3PCVBntNJkjBJd0
'' SIG '' EXeprmDK8HZrz8y/KSwEewbTMjneMRIRcJV2GMoAyUzK
'' SIG '' EyFJF7m6ByeoNBDDXMD4AQF5C2l3wfvRbREw+G+cyeya
'' SIG '' 564AHpFJ9sfqi9jAycnEZISxEf6N72GgGQr04cZt3JTb
'' SIG '' TktZGt6uWdZnseXDLGRHzFu4P4EqmVKvfMDliaKyWBzV
'' SIG '' lY+gmtE1vSxIK79WTySFrFY5j1vtbKyHybk6tv4TOmLt
'' SIG '' Lw4OP/ngVDiw/ZP+b7saGxPOEfuh5LOAr+llt+6V5NhT
'' SIG '' dkYPR6cbokawwX2DbGg7IeVuUPdf9bx/LznFV/8kPxCB
'' SIG '' J6qg0dqvCmQt/XbBXPJUvDoVJ9z+9hDlrAoLESqpHK8U
'' SIG '' 73UFGohBMJ/7itIyKngem7Ms98z6x0i63FJA8C06g+XE
'' SIG '' GJUmR28+c6ba+m/sc6xHI4zdL5a84UeRS9+u8UL8N4av
'' SIG '' fpSxHYTTxVjk1ZlwQ8dLARwEVWvTu4lRcEck3RPogx7o
'' SIG '' o8wUpSw9++m+Oi+K/z7RP/R67uCEtrkdM5N/xUoQR/t6
'' SIG '' 3P2ia9g8+LuYJ2BNBd1+qMcSoSlDfIFLFMeeqESbMv+j
'' SIG '' UsRQJn2lZvufAgMBAAGjggICMIIB/jAfBgNVHSMEGDAW
'' SIG '' gBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
'' SIG '' RPnrJ9yT26rMrd/Q4PfQSlgeaRcwPQYDVR0gBDYwNDAy
'' SIG '' BgVngQwBAzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
'' SIG '' LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
'' SIG '' MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0w
'' SIG '' gaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
'' SIG '' bS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNB
'' SIG '' NDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
'' SIG '' Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
'' SIG '' dGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
'' SIG '' Q0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBc
'' SIG '' BggrBgEFBQcwAoZQaHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
'' SIG '' bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0T
'' SIG '' BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAPIZ9C4IuKg85
'' SIG '' 0H8M66OLMVllIpANmEjijIG/D65qP6jfma+2WFps2bb+
'' SIG '' vKKIEtG9TFtCAlSrA1OV6Ru78paq0gqw/q3QAN0OleFR
'' SIG '' F1NvT/TLO0ulhjXTETB1BwSKZaJH001DH0+60TmbGfVh
'' SIG '' mwLrEj2Zfl4COFwivZ/2tRoknZl5tmiu4hpjDbOapWd9
'' SIG '' 5cqliSoMeu2i588557eikdiDk6bXBa5MxHjyXvbvaIWN
'' SIG '' Zjsymn0L0sYkF3BToDN0h0XiTTZjZPHdSV2GjX6C/+Nj
'' SIG '' HSZGRtBaIxdAL1mKEWJV9qkEKELPb098rO+zGdn1E5IH
'' SIG '' pS1zSRIYngwZEjZfyoCkg1F+EW4QPs2+ff2CsW5i3IEd
'' SIG '' 7vL19w9oyce3S4wejvnlTKt+EQWn3gSPxrbkp3BxdWRN
'' SIG '' 25MKo5bS5bJ21BiwqcIJHcTxWNgaQPh4sZboi9IimCSn
'' SIG '' UenPQhqdeOuz0NnmepMEYd7L/kqnAeUZ8bW1UbITAWQ/
'' SIG '' UT3phjCfRQNpEh4OBoUcH8hOOHZ0Lh86mxv9lDP3i7c/
'' SIG '' bQzZnrJr2dS34JXolwREbTX9Yu5JB5X3YP539/M1Ms5O
'' SIG '' FZou/MHQBrWMsVBsEXDTWqwbMTYw/zvIiHXcGUnpzInw
'' SIG '' 1jQXlLrmwVHW4ug+A4LFO7bWC4mnZoDNWp0xnlGmbP6I
'' SIG '' ccJmW5C7RJMxghqKMIIahgIBATB9MGkxCzAJBgNVBAYT
'' SIG '' AlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8G
'' SIG '' A1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNp
'' SIG '' Z25pbmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEA/f
'' SIG '' RMTG8AntqksfIadaPxAwDQYJYIZIAWUDBAIBBQCggZww
'' SIG '' GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
'' SIG '' BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcN
'' SIG '' AQkEMSIEIBEPGF7HS2ttBf6iUBn/aeeQtmPsSZZXyN1R
'' SIG '' momfMQLbMDAGCisGAQQBgjcCAQwxIjAgoB6AHABBAGQA
'' SIG '' bwBiAGUAIABJAG4ARABlAHMAaQBnAG4wDQYJKoZIhvcN
'' SIG '' AQEBBQAEggIAfiSp8H1Dd1yZ0mnt9MBAY5axvT7/YA3R
'' SIG '' Vovat8RqM+rnFAq4Tga5omZPGakIcb8f34WJ624LvPTU
'' SIG '' K00T2hFsgEHEgsg65Gk/GeR/ZLUMmc1TGHtl9Fu5qy5z
'' SIG '' QWMydrhO41/DYfP/rFvaS7gpXUaGNXaS8i0PiYToRfiC
'' SIG '' 3dHM8prd9jrpm3OPQRRIy/cgqxmLdWyp0EHiglg9Z87L
'' SIG '' 653D697ImPipaAyUMNOYzmnK/W4s0Aao5EeP7PMrmKNz
'' SIG '' cmNwOYb+uVl17oU+d4UcWQ4GPcSnvmuCdcRZhSzJMEk9
'' SIG '' /RLvUGBKolIbH67N101ZoQ+jA2xwKA6OasRSgHbHiFAU
'' SIG '' EXslCe+m7RF1Q4Rry3aH63fwL/D6AQ/RbfZ6uXWLe8dj
'' SIG '' 7e2DaP+vbDKjCDPiUrU4EbRD33mv2O6tUM+tFWF8GAaX
'' SIG '' wUk20ofWNEM1zqICB5WO5OC9EIitHcwOZbO/PSRT3VVD
'' SIG '' uIFO9YvEtnEgWzcxJTXaRQpo8Bf3HtSmN/NqE8+463lc
'' SIG '' emWChVTWbuK9co5McJjl7x64hxT3E2Dgm8n37cReMUNR
'' SIG '' VHxi2M/a1ogSBQsnhbIabSmFiLh/oNU8MIbWIalZYCJz
'' SIG '' P1NKdLN6d76PJem2gnDihA/m7VP5cQC//EAO3qZMlze6
'' SIG '' XVWRLvL4hy4ZPM/R9e0tc+exEUF76mEkPXShghc/MIIX
'' SIG '' OwYKKwYBBAGCNwMDATGCFyswghcnBgkqhkiG9w0BBwKg
'' SIG '' ghcYMIIXFAIBAzEPMA0GCWCGSAFlAwQCAQUAMHcGCyqG
'' SIG '' SIb3DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEwMTAN
'' SIG '' BglghkgBZQMEAgEFAAQgvujfTh6tt5Te5vvP/jZcOkCu
'' SIG '' ex4gJcOCb5zSMVdRoaYCECRFnNe95RgLh6OXgBA0b4YY
'' SIG '' DzIwMjQwMTI2MjM1NDQxWqCCEwkwggbCMIIEqqADAgEC
'' SIG '' AhAFRK/zlJ0IOaa/2z9f5WEWMA0GCSqGSIb3DQEBCwUA
'' SIG '' MGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2Vy
'' SIG '' dCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3Rl
'' SIG '' ZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcg
'' SIG '' Q0EwHhcNMjMwNzE0MDAwMDAwWhcNMzQxMDEzMjM1OTU5
'' SIG '' WjBIMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
'' SIG '' cnQsIEluYy4xIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVz
'' SIG '' dGFtcCAyMDIzMIICIjANBgkqhkiG9w0BAQEFAAOCAg8A
'' SIG '' MIICCgKCAgEAo1NFhx2DjlusPlSzI+DPn9fl0uddoQ4J
'' SIG '' 3C9Io5d6OyqcZ9xiFVjBqZMRp82qsmrdECmKHmJjadNY
'' SIG '' nDVxvzqX65RQjxwg6seaOy+WZuNp52n+W8PWKyAcwZeU
'' SIG '' tKVQgfLPywemMGjKg0La/H8JJJSkghraarrYO8pd3hkY
'' SIG '' hftF6g1hbJ3+cV7EBpo88MUueQ8bZlLjyNY+X9pD04T1
'' SIG '' 0Mf2SC1eRXWWdf7dEKEbg8G45lKVtUfXeCk5a+B4WZfj
'' SIG '' RCtK1ZXO7wgX6oJkTf8j48qG7rSkIWRw69XloNpjsy7p
'' SIG '' Be6q9iT1HbybHLK3X9/w7nZ9MZllR1WdSiQvrCuXvp/k
'' SIG '' /XtzPjLuUjT71Lvr1KAsNJvj3m5kGQc3AZEPHLVRzapM
'' SIG '' ZoOIaGK7vEEbeBlt5NkP4FhB+9ixLOFRr7StFQYU6mII
'' SIG '' E9NpHnxkTZ0P387RXoyqq1AVybPKvNfEO2hEo6U7Qv1z
'' SIG '' fe7dCv95NBB+plwKWEwAPoVpdceDZNZ1zY8SdlalJPrX
'' SIG '' xGshuugfNJgvOuprAbD3+yqG7HtSOKmYCaFxsmxxrz64
'' SIG '' b5bV4RAT/mFHCoz+8LbH1cfebCTwv0KCyqBxPZySkwS0
'' SIG '' aXAnDU+3tTbRyV8IpHCj7ArxES5k4MsiK8rxKBMhSVF+
'' SIG '' BmbTO77665E42FEHypS34lCh8zrTioPLQHsCAwEAAaOC
'' SIG '' AYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8E
'' SIG '' AjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1Ud
'' SIG '' IAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNV
'' SIG '' HSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNV
'' SIG '' HQ4EFgQUpbbvE+fvzdBkodVWqWUxo97V40kwWgYDVR0f
'' SIG '' BFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1
'' SIG '' NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEE
'' SIG '' gYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
'' SIG '' Z2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2Nh
'' SIG '' Y2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3Rl
'' SIG '' ZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNy
'' SIG '' dDANBgkqhkiG9w0BAQsFAAOCAgEAgRrW3qCptZgXvHCN
'' SIG '' T4o8aJzYJf/LLOTN6l0ikuyMIgKpuM+AqNnn48XtJoKK
'' SIG '' cS8Y3U623mzX4WCcK+3tPUiOuGu6fF29wmE3aEl3o+uQ
'' SIG '' qhLXJ4Xzjh6S2sJAOJ9dyKAuJXglnSoFeoQpmLZXeY/b
'' SIG '' JlYrsPOnvTcM2Jh2T1a5UsK2nTipgedtQVyMadG5K8TG
'' SIG '' e8+c+njikxp2oml101DkRBK+IA2eqUTQ+OVJdwhaIcW0
'' SIG '' z5iVGlS6ubzBaRm6zxbygzc0brBBJt3eWpdPM43UjXd9
'' SIG '' dUWhpVgmagNF3tlQtVCMr1a9TMXhRsUo063nQwBw3syY
'' SIG '' nhmJA+rUkTfvTVLzyWAhxFZH7doRS4wyw4jmWOK22z75
'' SIG '' X7BC1o/jF5HRqsBV44a/rCcsQdCaM0qoNtS5cpZ+l3k4
'' SIG '' SF/Kwtw9Mt911jZnWon49qfH5U81PAC9vpwqbHkB3NpE
'' SIG '' 5jreODsHXjlY9HxzMVWggBHLFAx+rrz+pOt5Zapo1iLK
'' SIG '' O+uagjVXKBbLafIymrLS2Dq4sUaGa7oX/cR3bBVsrquv
'' SIG '' czroSUa31X/MtjjA2Owc9bahuEMs305MfR5ocMB3CtQC
'' SIG '' 4Fxguyj/OOVSWtasFyIjTvTs0xf7UGv/B3cfcZdEQcm4
'' SIG '' RtNsMnxYL2dHZeUbc7aZ+WssBkbvQR7w8F/g29mtkIBE
'' SIG '' r4AQQYowggauMIIElqADAgECAhAHNje3JFR82Ees/Shm
'' SIG '' Kl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVT
'' SIG '' MRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsT
'' SIG '' EHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lD
'' SIG '' ZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAw
'' SIG '' MDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVT
'' SIG '' MRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UE
'' SIG '' AxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNI
'' SIG '' QTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3
'' SIG '' DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXH
'' SIG '' JQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI
'' SIG '' 82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR
'' SIG '' 8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU
'' SIG '' 5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3Hxq
'' SIG '' V3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtA
'' SIG '' rF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECn
'' SIG '' wHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu
'' SIG '' 9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz
'' SIG '' 9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpUR
'' SIG '' K1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD
'' SIG '' 4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/
'' SIG '' BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T
'' SIG '' /jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uK
'' SIG '' IqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11G
'' SIG '' deJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzS
'' SIG '' M7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB
'' SIG '' /wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+e
'' SIG '' yG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4c
'' SIG '' D08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsG
'' SIG '' AQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcw
'' SIG '' AYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsG
'' SIG '' AQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
'' SIG '' b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNV
'' SIG '' HR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2Vy
'' SIG '' dC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAg
'' SIG '' BgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEw
'' SIG '' DQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m
'' SIG '' 1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxp
'' SIG '' wc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGId
'' SIG '' DAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqr
'' SIG '' hc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp8
'' SIG '' 76i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeC
'' SIG '' RK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY
'' SIG '' +/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHx
'' SIG '' cpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fx
'' SIG '' ZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhz
'' SIG '' q6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o
'' SIG '' 08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1Oby
'' SIG '' F5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvt
'' SIG '' lUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ
'' SIG '' 8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8
'' SIG '' mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt
'' SIG '' 1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAY
'' SIG '' WjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2Vy
'' SIG '' dCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAw
'' SIG '' MDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJV
'' SIG '' UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
'' SIG '' ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
'' SIG '' Q2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
'' SIG '' DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAi
'' SIG '' MGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf
'' SIG '' 8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+
'' SIG '' n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
'' SIG '' bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7Fsa
'' SIG '' vOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbu
'' SIG '' yntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHE
'' SIG '' tWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6R
'' SIG '' AXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92k
'' SIG '' J7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI
'' SIG '' 8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppE
'' SIG '' GSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LV
'' SIG '' jHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QY
'' SIG '' uKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+N
'' SIG '' P8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtv
'' SIG '' sauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
'' SIG '' sh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB
'' SIG '' /zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08w
'' SIG '' HwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8w
'' SIG '' DgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
'' SIG '' BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQu
'' SIG '' Y29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5k
'' SIG '' aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
'' SIG '' Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9j
'' SIG '' cmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJ
'' SIG '' RFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0G
'' SIG '' CSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW
'' SIG '' 1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0
'' SIG '' E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n0
'' SIG '' 96wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE
'' SIG '' 1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLih
'' SIG '' Vo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsg
'' SIG '' jTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt
'' SIG '' 9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cB
'' SIG '' qZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQG
'' SIG '' EwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5
'' SIG '' BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5
'' SIG '' NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhAFRK/zlJ0I
'' SIG '' Oaa/2z9f5WEWMA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqG
'' SIG '' SIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0B
'' SIG '' CQUxDxcNMjQwMTI2MjM1NDQxWjArBgsqhkiG9w0BCRAC
'' SIG '' DDEcMBowGDAWBBRm8CsywsLJD4JdzqqKycZPGZzPQDAv
'' SIG '' BgkqhkiG9w0BCQQxIgQgTFYIpEthY/5EI8W7YGPx0vgx
'' SIG '' oWeDwWAp9HRaILW+418wNwYLKoZIhvcNAQkQAi8xKDAm
'' SIG '' MCQwIgQg0vbkbe10IszR1EBXaEE2b4KK2lWarjMWr00a
'' SIG '' mtQMeCgwDQYJKoZIhvcNAQEBBQAEggIAS8uB3MB+juxK
'' SIG '' bEh5wUKlkawBfAWOIoSnJqg2pUDG2tTPWDbgw2OAa7vC
'' SIG '' E6IFKX9mhY33dyYOA0DgPWOez02USP8nDjfOkjHtl/uU
'' SIG '' p0dlAQOrkC9nsWScsKY9e5xbF3j2R9qvqAcNGza87q0P
'' SIG '' vYxOrJaT6dJ9ZnDqktphOdqDVszRLuSr3pw3Hqd8wAsb
'' SIG '' lUl+bQgsgc1Al2vM3ZGP/YvZVPo2ZwwtskcgvTyiIYTQ
'' SIG '' c17Pzi9+UdN/HIi8BYqspiyLHVxEVq6/QIMMmVcM2WNU
'' SIG '' P8ROC/xMDrk0oUxcLKGhcZsYwNStpbTzywLPOkuGPY+c
'' SIG '' JoCmn28iec2p97FmjetFBV41Gh8EpNyH54kF1ujckho+
'' SIG '' dGqD9iXZc37TGGvZ4UBREzfPxBk/KUf01p5CaIuEf05L
'' SIG '' t9uh4I35Q3CB/2Kxt7qq0syJ8UeiIuoRDk91O6MI4XtU
'' SIG '' t5kIj5cB7wTTLfz6awO8EEhAPdamaF6tVi3I60bLuifg
'' SIG '' yfoxsqryhJOW0iS4IrxvrPmj4/A+d32AF3sgsix/DluX
'' SIG '' 2A03S+9viJw2l6cRMcBRVEW7FhMbJOpWf116v2kt0Erj
'' SIG '' FjlaqKqfUc6fQNWO5BpB51IPg25KSmpRe7eMfZ0AUGX8
'' SIG '' jqvBbq0JOvpVEorEVuqd0B0hJzTjJnqb+siP+NCjqx2P
'' SIG '' AeNJTCXVIjk=
'' SIG '' End signature block
